using System;
using System.Collections.Specialized;
using System.Xml;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary
{
	/// <summary>
	/// This class represents collection of DivisionInfo objects 
	/// accessed by division IEN. 
	/// </summary>
	public class DivisionInfoCollection : NameObjectCollectionBase
	{
		private DivisionInfo _defaultDivisionInfo = null;

		// Constants used in XML message serialization/deserialization
		private const string XMLCONSTS_DIVISIONS_NODE_NAME = "Divisions";

		/// <summary>
		/// Default constructor initializing the collection via base constructor. 
		/// </summary>
		public DivisionInfoCollection() 
			: base() {}

		/// <summary>
		/// Adds division info to the collection. Division info cannot be null. 
		/// It may be retrieved by division IEN.
		/// </summary>
		/// <param name="infoToAdd">Division info to add.</param>
		/// <param name="makeDefault">
		///		Flag indicating whether new division info 
		///		should be set default for the collection. 
		///		First division info added to the collection will be made default
		///		disregarding this paremeter.
		///	</param>
		/// <returns>Reference to added division info.</returns>
		public DivisionInfo Add( DivisionInfo infoToAdd, bool makeDefault )
		{
			if( infoToAdd == null )
				throw( new ArgumentNullException( "infoToAdd" ) );

			if( this.BaseGet( infoToAdd.Ien ) != null )
				throw( new ArgumentException( SR.Exceptions.DivisionInfoCollectionDuplicateIEN( infoToAdd.Ien ) ) );

			BaseAdd( infoToAdd.Ien, infoToAdd );

			if( _defaultDivisionInfo == null || makeDefault )
				_defaultDivisionInfo = infoToAdd;

			return infoToAdd;
		}

		/// <summary>
		/// Creates division info from the supplied parameters 
		/// and adds it to the collection. All parameters are required.
		/// </summary>
		/// <param name="ien">Division IEN.</param>
		/// <param name="name">Division name.</param>
		/// <param name="number">Division number.</param>
		/// <param name="makeDefault">
		///		Flag indicating whether new division info 
		///		should be set default for the collection. 
		///		First division info added to the collection will be made default
		///		disregarding this paremeter.
		///	</param>
		/// <returns>Reference to added division.</returns>
		public DivisionInfo Add( string ien, string name, string number, bool makeDefault )
		{
			DivisionInfo _newInfo = new DivisionInfo( ien, name, number );

			return this.Add( _newInfo, makeDefault );
		}

		/// <summary>
		/// XML serialization method writing XML representation of class
		/// into supplied XmlWriter.
		/// </summary>
		/// <param name="writer">XmlWriter to use.</param>
		public virtual void WriteDivisionsListToXml( XmlWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			writer.WriteStartElement( XMLCONSTS_DIVISIONS_NODE_NAME );

			for( int i = 0; i < this.Count; i++ )
				this[i].WriteDivisionInfoToXml( writer, this[i] == this.DefaultDivisionInfo );

			writer.WriteEndElement();
		}

		/// <summary>
		/// XML deserialization factory method creating an instance of class from supplied parent XML element.
		/// Parent XML element must contain one and only one instance of serialized class among its children. 
		/// </summary>
		/// <param name="parentElement">
		///		Parent XML element to parse. It must contain one and 
		///		only one instance of serialized class among its children. 
		///	</param>
		/// <returns>Instance of class deserialized from a given parent XmlElement.</returns>
		public static DivisionInfoCollection Parse( XmlElement parentElement )
		{
			if( parentElement == null )
				throw( new ArgumentNullException( "parentElement" ) );

			XmlElement _divisionsElement = XmlUtility.ParseGetRequiredElementByUniqueTagName( parentElement, XMLCONSTS_DIVISIONS_NODE_NAME );			
			
			DivisionInfoCollection _resultDivisions = new DivisionInfoCollection();
			
			foreach( XmlNode _divNode in XmlUtility.ParseGetChildElements( _divisionsElement ) )
			{				
				try
				{
					bool _makeDefault = false;
					_resultDivisions.Add( DivisionInfo.Parse( _divNode, out _makeDefault ), _makeDefault );
				}
				catch( ArgumentException xcp ) // there may be duplicate division IENs 
				{
					throw( new XmlParseException( 
						SR.Exceptions.XmlDeserializationFailedDueToInnerException( typeof(DivisionInfoCollection).Name, _divNode.Name ), xcp ) );
				}
			}
			
			return _resultDivisions;
		}

		/// <summary>
		/// XML parsing utility method used for recognition during object deserialization. 
		/// It's used to test if a given element may be used as a parameter for
		/// the class's deserialization constructor. However, this method does not guarantee, 
		/// that no exception will be thrown during deserialization. It only estimates. 
		/// </summary>
		/// <param name="parentElement">XmlElement to test.</param>
		/// <returns>True if a given element contains XML needed to deserialize class from it.</returns>
		public static bool XmlParseIsValidParentElement( XmlElement parentElement )
		{
			if( parentElement == null )
				throw( new ArgumentNullException( "parentElement" ) );

			return parentElement.GetElementsByTagName( XMLCONSTS_DIVISIONS_NODE_NAME ).Count == 1;
		}

		/// <summary>
		/// Retrieves division info by integer index. 
		/// </summary>
		public DivisionInfo this[int index]
		{
			get
			{
				try
				{
					return (DivisionInfo)BaseGet( index );
				}
				catch( ArgumentOutOfRangeException xcp )
				{
					throw( new IndexOutOfRangeException( xcp.Message ) );
				}
			}
		}

		/// <summary>
		/// Retrieves division info by division IEN. 
		/// </summary>
		public DivisionInfo GetDivisionByIen( string ienKey )
		{
			return (DivisionInfo)BaseGet( ienKey );
		}

		/// <summary>
		/// Default division info. If no division infos are available 
		/// in the collection null will be returned. First added division info will be marked as default 
		/// disregarding the value of makeDefault paremeter. 
		/// </summary>
		public DivisionInfo DefaultDivisionInfo
		{
			get
			{
				return _defaultDivisionInfo;
			}
		}
	}
}
